<?php
require_once 'config.php';

function json($data){
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

/* ================= API ================= */
if(isset($_GET['action'])){

    if($_GET['action']=='teams'){
        $r=mysqli_query($conn,"SELECT id,name FROM teams ORDER BY name");
        json(mysqli_fetch_all($r,MYSQLI_ASSOC));
    }

    // baru: daftar kegiatan unik (opsional filter berdasarkan tim/bulan/tahun)
    if($_GET['action']=='activities'){
        $team_id = $_GET['team_id'] ?? '';
        $month   = $_GET['month'] ?? '';
        $year    = $_GET['year'] ?? '';

        $where = [];
        if($team_id) $where[] = "am.team_id=".intval($team_id);
        if($month) $where[] = "am.month='".intval($month)."'";
        if($year)  $where[] = "YEAR(am.deadline)=".intval($year);

        $whereSQL = $where ? "WHERE ".implode(" AND ",$where) : "";

        $q = "SELECT DISTINCT am.activity_name FROM activity_master am $whereSQL ORDER BY am.activity_name";
        $r = mysqli_query($conn,$q);
        $list = [];
        while($row = mysqli_fetch_assoc($r)){
            if($row['activity_name'] !== null && $row['activity_name'] !== '') $list[] = $row['activity_name'];
        }
        json($list);
    }

    if($_GET['action']=='data'){
        $team_id = $_GET['team_id'] ?? '';
        $month   = $_GET['month'] ?? '';
        $year    = $_GET['year'] ?? '';
        $activity = $_GET['activity'] ?? '';

        $where = [];
        if($team_id) $where[] = "am.team_id=".intval($team_id);
        if($month) $where[] = "am.month='".intval($month)."'";
        if($year)  $where[] = "YEAR(am.deadline)=".intval($year); // filter tahun berdasarkan kolom deadline
        if($activity) $where[] = "am.activity_name='".mysqli_real_escape_string($conn, $activity)."'";
        $whereSQL = $where ? "WHERE ".implode(" AND ",$where) : "";

        $q = "
        SELECT 
            am.team_id, t.name team,
            am.activity_name,
            SUM(am.target_volume) AS target_total,
            SUM(IFNULL(de.volume_today,0)) AS realisasi_total,
            MAX(am.deadline) AS deadline
        FROM activity_master am
        LEFT JOIN daily_entries de ON de.activity_master_id=am.id
        LEFT JOIN teams t ON t.id=am.team_id
        $whereSQL
        GROUP BY am.team_id, am.activity_name
        ORDER BY t.name, am.activity_name
        ";
        $r = mysqli_query($conn,$q);
        $data = [];
        $totalTarget = 0;
        $totalRealisasi = 0;
        $deadlines = [];

        while($row = mysqli_fetch_assoc($r)){
            $row['capaian'] = $row['target_total'] ? round($row['realisasi_total']/$row['target_total']*100,0) : 0;
            $data[] = $row;
            $totalTarget += $row['target_total'];
            $totalRealisasi += $row['realisasi_total'];
            if($row['deadline']) $deadlines[] = $row['deadline'];
        }

        $capaianTotal = $totalTarget ? round($totalRealisasi/$totalTarget*100,0) : 0;
        $sisaHari = '-';
        $nearest_deadline = null;

        // ============================
        // BARU: cari activity dengan deadline paling mendekat (upcoming, tanggal minimum > hari ini)
        // ============================
        $nearest_activity = null;
        if($data){
            $today = new DateTimeImmutable();
            $minDt = null;
            foreach($data as $row){
                if(empty($row['deadline'])) continue;
                try{
                    $dt = new DateTimeImmutable($row['deadline']);
                }catch(Exception $e){
                    continue;
                }
                // hanya ambil deadline yang masih di masa depan (strict > today)
                if($dt > $today){
                    if($minDt === null || $dt < $minDt){
                        $minDt = $dt;
                        $nearest_activity = [
                            'team' => $row['team'],
                            'activity_name' => $row['activity_name'],
                            'deadline' => $dt->format('Y-m-d')
                        ];
                    }
                }
            }
        }

        // Perilaku sebelumnya: hitung sisaHari dari deadline terbaru untuk ringkasan.
        // Kita pertahankan logika 'sisaHari' seperti semula untuk kompatibilitas.
        $sisaHari = '-';
        $nearest_deadline = null;
        if(!empty($deadlines)){
            // tetap gunakan logic "tanggal terbesar" untuk ringkasan.sisaHari (tidak diubah)
            $max = null;
            foreach($deadlines as $d){
                try{
                    $dt = new DateTimeImmutable($d);
                }catch(Exception $e){
                    continue;
                }
                if($max === null || $dt > $max){
                    $max = $dt;
                }
            }
            if($max){
                $nearest_deadline = $max->format('Y-m-d');
                $today = new DateTimeImmutable();
                $diff = $today->diff($max);
                $sisaHari = max(0, $diff->days);
            }
        }

        // jika ada nearest_activity (upcoming), hitung sisaHari khusus untuk activity tersebut
        if($nearest_activity){
            try{
                $dt = new DateTimeImmutable($nearest_activity['deadline']);
                $today = new DateTimeImmutable();
                $diff = $today->diff($dt);
                $nearest_activity['sisaHari'] = max(0, $diff->days);
            }catch(Exception $e){
                $nearest_activity['sisaHari'] = '-';
            }
        }

        $q2 = "
        SELECT e.name employee,
            SUM(am.target_volume) AS target_total,
            SUM(IFNULL(de.volume_today,0)) AS realisasi_total,
            MAX(am.deadline) AS deadline
        FROM activity_master am
        LEFT JOIN daily_entries de ON de.activity_master_id=am.id
        LEFT JOIN employees e ON e.id=am.employee_id
        ".($where ? "WHERE ".implode(" AND ",$where) : "")."
        GROUP BY e.id
        ORDER BY e.name
        ";
        $r2 = mysqli_query($conn,$q2);
        $rekapPegawai = [];
        while($row = mysqli_fetch_assoc($r2)){
            $row['capaian'] = $row['target_total'] ? round($row['realisasi_total']/$row['target_total']*100,0) : 0;
            $row['sisaHari'] = $row['deadline'] ? max(0,ceil((new DateTime($row['deadline']))->diff(new DateTime())->days)) : '-';
            $rekapPegawai[] = $row;
        }

        json([
            'data' => $data,
            'summary' => [
                'totalTarget'=>$totalTarget,
                'totalRealisasi'=>$totalRealisasi,
                'capaianTotal'=>$capaianTotal,
                'sisaHari'=>$sisaHari,
                'nearest_deadline'=>$nearest_deadline // tetap untuk kompatibilitas (tanggal terbaru)
            ],
            'nearest_activity' => $nearest_activity, // BARU: activity yang paling mendekati deadline (upcoming)
            'rekapPegawai' => $rekapPegawai
        ]);
    }

    if($_GET['action']=='pie_team'){
        $month = $_GET['month'] ?? '';
        $year  = $_GET['year'] ?? '';
        $activity = $_GET['activity'] ?? '' ;

        $where = [];
        if($month) $where[] = "am.month='".intval($month)."'";
        if($year)  $where[] = "YEAR(am.deadline)=".intval($year);
        if($activity) $where[] = "am.activity_name='".mysqli_real_escape_string($conn, $activity)."'" ;

        $whereSQL = $where ? "WHERE ".implode(" AND ", $where) : "";

        $q = "
        SELECT 
            t.name team,
            SUM(am.target_volume) AS target_total,
            SUM(IFNULL(de.volume_today,0)) AS realisasi_total
        FROM teams t
        LEFT JOIN activity_master am ON am.team_id=t.id
        LEFT JOIN daily_entries de ON de.activity_master_id=am.id
        $whereSQL
        GROUP BY t.id
        ORDER BY t.name
        ";
        $r = mysqli_query($conn,$q);
        json(mysqli_fetch_all($r,MYSQLI_ASSOC));
    }
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
<meta charset="UTF-8">
<title>Dashboard Kegiatan</title>
<meta name="viewport" content="width=device-width, initial-scale=1">
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script src="https://cdn.jsdelivr.net/npm/chartjs-plugin-datalabels@2.2.0"></script>

<style>
/* ---------- UI Kompak / Efisien (dimodifikasi) ---------- */

/* Variabel dasar */
:root{
  --ui-font: Inter, system-ui, -apple-system, "Segoe UI", Roboto, "Helvetica Neue", Arial;
  --base-font-size: 13px;
  --muted: #6b7280;
  --card-radius: 8px;
  --card-padding: 8px;
}
body{
    font-family: var(--ui-font);
    font-size: var(--base-font-size);
    color:#0f172a;
    background:#f3f4f6;
    margin:0;
}
.container{
  width:100%;
  max-width: min(3840px, 98vw);
  padding-left:12px;
  padding-right:12px;
}

/* Kartu & spasi kompak */
.card{
    background:#fffffff3;
    border-radius: var(--card-radius);
    padding: calc(var(--card-padding) - 2px);
    box-shadow: 0 6px 18px rgba(15,23,42,0.04);
    margin-bottom:10px;
}

/* Heading lebih kecil */
h4{ font-size:16px; margin-bottom:10px; font-weight:600; color:#0b1220; }
h6{ font-size:13px; margin-bottom:8px; font-weight:600; color:#081022; }

/* Kontrol form - kompak */
.form-control{
    height:34px;
    padding:6px 8px;
    font-size:13px;
    border-radius:6px;
    border:1px solid #d1d5db;
    background: #fff;
}

/* Baris filter kompak */
.filter-toggle { gap:8px; display:flex; align-items:center; justify-content:space-between; }
.filter-toggle .left { display:flex; gap:8px; align-items:center; }
.filter-toggle small { font-size:11px; color:var(--muted); }

/* Tombol kompak */
.btn{ font-size:13px; padding:6px 9px; border-radius:6px; }

/* Wrapper chart - padding lebih kecil */
.chart-wrapper{ padding-bottom:6px; overflow:hidden; }

/* Canvas responsif */
#chart, #chartPegawai, #pieTeam{ display:block; width:100%; height:auto; }

#chart .chartjs-render-monitor { cursor: default; }

/* Ringkasan kompak (sekarang satu bar) */
.summary-bar { display:flex; gap:8px; align-items:stretch; }
.summary-item {
    flex:1;
    min-width:0;
    padding:12px 10px;
    border-radius:8px;
    color:#fff;
    text-align:left;
    display:flex;
    align-items:center;
    gap:10px;
}
.summary-left {
    display:flex;
    flex-direction:column;
    gap:4px;
    min-width:0;
}
.summary-icon {
    width:44px; height:44px; border-radius:8px; display:flex; align-items:center; justify-content:center;
    background: rgba(255,255,255,0.12); font-size:18px;
}
.summary-label { font-size:12px; opacity:.9; color:rgb(255, 255, 255); }
.summary-value { font-weight:800; font-size:1.5rem; line-height:1; color:#fff; }

/* WARNA GRADIENT */
.bg-blue{
  background: linear-gradient(135deg, #3b82f6 0%, #1e40af 100%);
  color:#fff;
}

.bg-green{
  background: linear-gradient(135deg, #10b981 0%, #047857 100%);
  color:#fff;
}

.bg-yellow{
  background: linear-gradient(135deg, #facc15 0%, #ca8a04 100%);
  color:#111;
}

.bg-gray{
  background: linear-gradient(135deg, #6b7280 0%, #374151 100%);
  color:#fff;
}


.small-note{ font-size:12px; color:var(--muted); }

 /* ---------- Gaya Podium (BARU) ---------- */
.podium-wrap { padding:8px; }
.podium-grid {
    display:flex;
    align-items:flex-end;
    justify-content:center;
    gap:12px;
    padding:12px 6px 6px 6px;
    height:200px;
}
.podium-item {
    width:80px;
    display:flex;
    flex-direction:column;
    align-items:center;
    justify-content:flex-end;
    gap:6px;
}
.podium-bar {
    width:100%;
    border-radius:6px 6px 0 0;
    display:flex;
    align-items:center;
    justify-content:center;
    color:#fff;
    font-weight:800;
    font-size:20px;
    box-shadow: 0 6px 12px rgba(2,6,23,0.08);
    position:relative;
}

.podium-bar.pos1 { background: linear-gradient(135deg,#f59e0b,#d97706); }
.podium-bar.pos2 { background: linear-gradient(135deg,#60a5fa,#2563eb); }
.podium-bar.pos3 { background: linear-gradient(135deg,#34d399,#059669); }
.podium-bar.pos4 { background: linear-gradient(135deg,#a78bfa,#7c3aed); }
.podium-bar.pos5 { background: linear-gradient(135deg,#f97316,#ea580c); }

.podium-pos { font-weight:800; font-size:12px; margin-top:6px; color:#111; }
.podium-name { font-size:13px; color:#081022; white-space:nowrap; overflow:hidden; text-overflow:ellipsis; max-width:110px; text-align:center; }
.podium-percent { font-size:12px; color:#081022; opacity:.8; }

@media (max-width:991.98px){
    :root{ --base-font-size:12px; }
    .summary-bar{ flex-wrap:wrap; gap:8px; }
    .summary-item{ flex:1 1 48%; }
    #chart, #pieTeam, #chartPegawai { min-height:240px !important; }
    .podium-grid{ height:150px; gap:8px; }
    .podium-item{ width:52px; }
    .podium-bar { font-size:16px; }
    .podium-name { font-size:11px; max-width:80px;}
}

@media (min-width:992px){
    .summary-item{ flex:1; }
    #chart{ min-height:260px; } /* sedikit lebih besar untuk label di atas bar */
    #chartPegawai{ min-height:220px; }
    #pieTeam{ min-height:220px; }
}

.compact .card{ padding:6px; border-radius:6px; }
.compact h6{ font-size:12px; }

.deadline-card { background: linear-gradient(135deg,#dc2626,#ff4d4d); color:#fff; border-radius:8px; padding:10px; }
.deadline-title { display:flex; align-items:center; gap:8px; font-weight:700; font-size:13px; }
.deadline-body { font-size:13px; margin-top:6px; color:#fff; }

.chartjs-datalabels { font-weight:700; }

/* kecil: buat judul opsi menunjukkan kursor pointer saat hover pada select untuk aksesibilitas */
select[title] { cursor: pointer; }
</style>
</head>
<body>
<div class="container mt-3">

    <h4 class="mb-2"><i class="bi bi-speedometer2"></i> Dashboard Kegiatan Pegawai</h4>

    <!-- FILTER / FORM -->
    <div class="card mb-2">
        <div class="filter-toggle">
            <div class="left">
                <i class="bi bi-funnel-fill" style="font-size:16px;color:var(--muted)"></i>
                <strong style="font-size:14px">Filters</strong>
                <small class="small-note d-none d-md-inline">Ubah filter untuk memperbarui dashboard</small>
            </div>
            <!-- tombol Filter dihilangkan karena otomatis -->
        </div>

       <div class="mt-2" id="filtersCollapse">
    <form id="filterForm" class="row g-2 align-items-center">

        <!-- TIM -->
        <div class="col-12 col-md-2">
            <div class="input-group input-group-sm">
                <span class="input-group-text">
                    <i class="fa-solid fa-users"></i>
                </span>
                <select id="team" class="form-control">
                    <option value="">Semua Tim</option>
                </select>
            </div>
        </div>

        <!-- KEGIATAN -->
        <div class="col-12 col-md-3">
            <div class="input-group input-group-sm">
                <span class="input-group-text">
                    <i class="fa-solid fa-list-check"></i>
                </span>
                <select id="activity" class="form-control">
                    <option value="">Semua Kegiatan</option>
                </select>
            </div>
        </div>

        <!-- BULAN -->
        <div class="col-6 col-md-2">
            <div class="input-group input-group-sm">
                <span class="input-group-text">
                    <i class="fa-solid fa-calendar-days"></i>
                </span>
                <select id="month" class="form-control">
                    <option value="">Semua Bulan</option>
                    <?php
                    $bulan = [
                        '01'=>'Januari','02'=>'Februari','03'=>'Maret','04'=>'April','05'=>'Mei','06'=>'Juni',
                        '07'=>'Juli','08'=>'Agustus','09'=>'September','10'=>'Oktober','11'=>'November','12'=>'Desember'
                    ];
                    $curMonth = date('m');
                    foreach ($bulan as $k => $v) {
                        $selected = ($k == $curMonth) ? 'selected' : '';
                        echo "<option value='$k' $selected>$v</option>";
                    }
                    ?>
                </select>
            </div>
        </div>

        <!-- TAHUN -->
        <div class="col-6 col-md-2">
            <div class="input-group input-group-sm">
                <span class="input-group-text">
                    <i class="fa-solid fa-calendar"></i>
                </span>
                <select id="year" class="form-control">
                    <option value="">Tahun</option>
                    <?php
                    $curYear = date('Y');
                    for ($y = $curYear; $y <= $curYear + 2; $y++){
                        $sel = ($y == $curYear) ? 'selected' : '';
                        echo "<option value='$y' $sel>$y</option>";
                    }
                    ?>
                </select>
            </div>
        </div>

        <!-- tombol apply dihapus karena filter otomatis -->

    </form>
</div>
        <!-- SUMMARY BAR: dipindahkan di bawah filter, satu bar -->
        <div class="mt-2">
            <div class="summary-bar">
                <div class="summary-item bg-blue" id="summaryTargetCard">
                    <div class="summary-left">
                        <div class="summary-icon"><i class="bi bi-bullseye"></i></div>
                        <div class="summary-label">Total Target</div>
                    </div>
                    <div style="margin-left:auto; text-align:right;">
                        <div id="totalTarget" class="summary-value">0</div>
                    </div>
                </div>
                <div class="summary-item bg-green" id="summaryRealisasiCard">
                    <div class="summary-left">
                        <div class="summary-icon"><i class="bi bi-bar-chart-line"></i></div>
                        <div class="summary-label">Total Realisasi</div>
                    </div>
                    <div style="margin-left:auto; text-align:right;">
                        <div id="totalRealisasi" class="summary-value">0</div>
                    </div>
                </div>
                <div class="summary-item bg-yellow" id="summaryCapaianCard" style="color:#111">
                    <div class="summary-left">
                        <div class="summary-icon"><i class="bi bi-percent"></i></div>
                        <div class="summary-label">Capaian %</div>
                    </div>
                    <div style="margin-left:auto; text-align:right;">
                        <div id="capaianTotal" class="summary-value">0%</div>
                    </div>
                </div>
                <div class="summary-item bg-gray" id="summarySisaCard">
                    <div class="summary-left">
                        <div class="summary-icon"><i class="bi bi-clock"></i></div>
                        <div class="summary-label">Sisa Hari</div>
                    </div>
                    <div style="margin-left:auto; text-align:right;">
                        <div id="sisaHari" class="summary-value">-</div>
                        <div id="sisaHariNote" class="small-note mt-1" style="color:#fff;opacity:.9; font-size:12px"></div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- AREA DASHBOARD -->
    <div class="row">
        <div class="col-lg-8">
            <div class="card mb-2">
                <h6 class="mb-1">Target Vs Realisasi Kegiatan</h6>
                <div class="chart-wrapper">
                    <canvas id="chart"></canvas>
                </div>
            </div>

            <div class="card mb-2">
                <h6 class="mb-1">Realisasi Vs Deadline Sisa Hari</h6>
                <div class="chart-wrapper">
                    <canvas id="chartPegawai"></canvas>
                </div>
            </div>
        </div>

        <div class="col-lg-4">
            <div class="card mb-2 text-center">
                <h6 class="mb-1">Total Capaian Tim %</h6>
                <div class="chart-wrapper" style="max-height:260px;">
                    <canvas id="pieTeam"></canvas>
                </div>
            </div>

            <div class="card mb-2 deadline-card">
                <div class="deadline-title">
                    <i class="bi bi-exclamation-triangle-fill"></i>
                    <div>Kegiatan Mendekati Deadline</div>
                </div>
                <div id="nearestActivity" class="deadline-body">-</div>
                <div id="nearestActivityNote" class="deadline-body small-note mt-1"></div>
            </div>

            <!-- BARU: Podium Top Performance (1-5) -->
            <div class="card mb-2">
                <h6 class="mb-1">Top 5 Capaian Realisasi Terbaik</h6>
                <div class="podium-wrap">
                    <div id="podium" class="podium-grid">
                        <!-- JS akan mengisi konten podium di sini -->
                    </div>
                </div>
            </div>

        </div>
    </div>

</div>

<!-- Bootstrap JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

<script>
/* registrasi plugin datalabels */
if(window.Chart && window.ChartDataLabels){ Chart.register(ChartDataLabels); }

let chart, chartPegawai, pieTeam;
let countdownInterval = null, chartAnimateInterval = null;

/* Referensi DOM (tombol apply dihapus) */
const team = document.getElementById('team');
const activity = document.getElementById('activity');
const month = document.getElementById('month');
const year = document.getElementById('year');

/* fungsi bantu */
function encodeHTML(s){
    if(!s && s !== 0) return '';
    return String(s).replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;')
                    .replace(/"/g,'&quot;').replace(/'/g,'&#39;');
}
function numberWithSep(x){
    if(x === null || x === undefined) return '0';
    return (Number(x).toLocaleString('id-ID'));
}
function animateNumber(el, end, suffix='') {
    let start = parseInt(String(el.innerText).replace(/\D/g,'')) || 0;
    let startTime = performance.now();
    function step(t){
        let p = Math.min(1,(t-startTime)/600);
        let val = Math.round(start+(end-start)*p);
        el.innerText = numberWithSep(val) + suffix;
        if(p<1) requestAnimationFrame(step);
    }
    requestAnimationFrame(step);
}

/* TRUNCATE helper: potong ke spasi terdekat sebelum limit jika memungkinkan */
function truncateText(s, n){
    if(!s && s !== 0) return '';
    s = String(s).trim();
    n = Math.max(8, (n || 28)); // min 8 chars
    if(s.length <= n) return s;
    // cari spasi terakhir sebelum batas n
    let idx = s.lastIndexOf(' ', n-1);
    if(idx >= 6) {
        return s.slice(0, idx) + '…';
    }
    return s.slice(0, n-1) + '…';
}

/* muat kontrol */
async function loadTeams(){
    try{
        let r = await fetch('dashboard.php?action=teams');
        let j = await r.json();
        team.innerHTML = '<option value="">Semua Tim</option>';
        j.forEach(x=>team.innerHTML+=`<option value="${x.id}">${encodeHTML(x.name)}</option>`);
    }catch(e){ console.error('loadTeams',e); }
}
async function loadActivities(){
    try{
        let url = `dashboard.php?action=activities&team_id=${team.value}&month=${month.value}&year=${year.value}`;
        let r = await fetch(url);
        let j = await r.json();
        activity.innerHTML = '<option value="">Semua Kegiatan</option>';
        j.forEach(x=>{
            // tampilkan teks terpotong, simpan nama lengkap di value dan title untuk hover
            const full = x || '';
            const short = truncateText(full, 36);
            activity.innerHTML += `<option value="${encodeHTML(full)}" title="${encodeHTML(full)}">${encodeHTML(short)}</option>`;
        });
    }catch(e){ console.error('loadActivities', e); }
}

/* event: filter otomatis saat terjadi perubahan */
team.onchange = function(){ loadActivities(); loadDashboard(); };
month.onchange = function(){ loadActivities(); loadDashboard(); };
year.onchange = function(){ loadActivities(); loadDashboard(); };
activity.onchange = loadDashboard;

/* pemuat dashboard */
async function loadDashboard(){
    try{
        let params = `team_id=${team.value}&month=${month.value}&year=${year.value}&activity=${encodeURIComponent(activity.value)}`;
        let r = await fetch(`dashboard.php?action=data&${params}`);
        let j = await r.json();

        // animasi ringkasan
        animateNumber(document.getElementById('totalTarget'), j.summary.totalTarget || 0);
        animateNumber(document.getElementById('totalRealisasi'), j.summary.totalRealisasi || 0);
        // capaian sebagai persen (integer)
        let capa = j.summary.capaianTotal || 0;
        document.getElementById('capaianTotal').innerText = (capa) + '%';

        startCountdown(j.summary.sisaHari, j.summary.nearest_deadline);
        renderNearestActivity(j.nearest_activity);

        drawChart(j.data);
        drawChartPegawai(j.rekapPegawai);
        await drawPieTeam();

        // BARU: render podium (top 1-5) berdasarkan kecepatan
        renderPodium(j.rekapPegawai);

        if(chartAnimateInterval) clearInterval(chartAnimateInterval);
        animateCharts();
        chartAnimateInterval = setInterval(animateCharts, 10000);
    }catch(e){ console.error('loadDashboard', e); }
}

/* drawChart: angka di dalam bagian atas bar, sembunyikan otomatis jika terlalu kecil */
const chartEl = document.getElementById('chart');

/* helper: kontras warna teks */
function getContrastColor(hex){
    if(!hex) return '#fff';
    hex = hex.replace('#','');
    if(hex.length === 3) hex = hex.split('').map(ch => ch+ch).join('');
    const r = parseInt(hex.substring(0,2),16);
    const g = parseInt(hex.substring(2,4),16);
    const b = parseInt(hex.substring(4,6),16);
    const lum = (0.299 * r + 0.587 * g + 0.114 * b) / 255;
    return lum > 0.6 ? '#111' : '#fff';
}

/* helper: gradient biru */
function blueGradient(ctx){
    const g = ctx.createLinearGradient(0, 0, 0, ctx.canvas.height);
    g.addColorStop(0, '#60a5fa');  // biru terang
    g.addColorStop(1, '#2563eb');  // biru gelap
    return g;
}

/* helper: gradient hijau */
function greenGradient(ctx){
    const g = ctx.createLinearGradient(0, 0, 0, ctx.canvas.height);
    g.addColorStop(0, '#34d399');  // hijau terang
    g.addColorStop(1, '#059669');  // hijau gelap
    return g;
}

function drawChart(data){
    // label lengkap (untuk tooltip), label pendek untuk sumbu
    const fullLabels = data.map(d=>d.activity_name || '');
    const labelsShort = fullLabels.map(l => truncateText(l, 28));

    const target = data.map(d=>Number(d.target_total)||0);
    const real   = data.map(d=>Number(d.realisasi_total)||0);

    const wrapperWidth = chartEl.parentElement.clientWidth || window.innerWidth * 0.66;
    const reserved = 50;
    const available = Math.max(260, wrapperWidth - reserved);
    const perBar = Math.max(6, Math.floor(available / Math.max(1, labelsShort.length)));
    const barThickness = Math.min(18, Math.max(5, Math.floor(perBar * 0.50)));

    const yMax = Math.max(1, ...target, ...real);
    const suggestedMax = Math.ceil(yMax * 1.08);
    const hideCutoff = yMax * 0.08;

    chartEl.style.width = '100%';

    if(chart) chart.destroy();

    // tangkap fullLabels untuk tooltip
    const fullLabelsLocal = fullLabels;

    chart = new Chart(chartEl, {
        type: 'bar',
        data: {
            labels: labelsShort,
            datasets: [
                {
                    label: 'Target',
                    data: target,
                    barThickness,
                    borderRadius: 1,
                    backgroundColor: ctx => blueGradient(ctx.chart.ctx),
                    datalabels: {
                        display: ctx => {
                            const v = ctx.dataset.data[ctx.dataIndex] || 0;
                            return v >= hideCutoff;
                        },
                        anchor: 'end',
                        align: 'start',
                        offset: -1,
                        formatter: v => numberWithSep(v),
                        font: { weight: '400', size: 10 },
                        color: '#fff'
                    }
                },
                {
                    label: 'Realisasi',
                    data: real,
                    barThickness,
                    borderRadius: 1,
                    backgroundColor: ctx => greenGradient(ctx.chart.ctx),
                    datalabels: {
                        display: ctx => {
                            const v = ctx.dataset.data[ctx.dataIndex] || 0;
                            return v >= hideCutoff;
                        },
                        anchor: 'end',
                        align: 'start',
                        offset: -1,
                        formatter: v => numberWithSep(v),
                        font: { weight: '400', size: 10 },
                        color: '#fff'
                    }
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'bottom',
                    labels: { boxWidth: 10, font: { size: 10 } }
                },
                tooltip: {
                    bodyFont: { size: 8 },
                    callbacks: {
                        // tampilkan nama kegiatan lengkap sebagai title saat hover
                        title: function(tooltipItems) {
                            if(!tooltipItems || !tooltipItems.length) return '';
                            const idx = tooltipItems[0].dataIndex;
                            return fullLabelsLocal[idx] || tooltipItems[0].label || '';
                        }
                    }
                },
                datalabels: { display: false }
            },
            scales: {
                x: {
                    ticks: {
                        autoSkip: true,
                        maxRotation: 60,
                        minRotation: 30,
                        font: { size: 9 }
                    },
                    grid: { display: false }
                },
                y: {
                    beginAtZero: true,
                    suggestedMax,
                    ticks: { precision: 0, font: { size: 8 } },
                    grid: {
                        borderDash: [4, 4],     // 🔹 garis putus-putus
                        color: 'rgba(148,163,184,.35)', // abu lembut
                        drawBorder: false
                    }
                }
            },
            animation: {
                duration: 600,
                easing: 'easeOutQuart'
            }
        }
    });
}


/* drawChartPegawai: chart untuk tiap pegawai */
const chartPegawaiEl = document.getElementById('chartPegawai');

function drawChartPegawai(data, totalHariKerja){
    const labels = data.map(x => x.employee);
    const realisasi = data.map(x => x.capaian);
    const sisaHari = data.map(x => (isNaN(x.sisaHari) ? 0 : Number(x.sisaHari)));
    totalHariKerja = totalHariKerja || 30;

    const warnaRealisasi = data.map(x=>{
        let hariTerpakai = totalHariKerja - (isNaN(x.sisaHari) ? totalHariKerja : x.sisaHari);
        let progressWaktu = (hariTerpakai/totalHariKerja)*100;
        return x.capaian < progressWaktu ? '#ef4444' : '#2bb5a9';
    });

    const perItemPx = 34;
    const minHeight = 120;
    const totalHeight = Math.max(minHeight, Math.min(480, labels.length * perItemPx));
    chartPegawaiEl.style.height = totalHeight + 'px';
    chartPegawaiEl.style.width = '100%';

    if(chartPegawai) chartPegawai.destroy();

    chartPegawai = new Chart(chartPegawaiEl, {
        type: 'bar',
        data: {
            labels,
            datasets: [
                {
                    label:'Realisasi (%)',
                    data: realisasi,
                    backgroundColor: warnaRealisasi,
                    barThickness: 10,
                    borderRadius: 1,
                    datalabels:{
                        anchor:'end',
                        align:'right',
                        offset:1,
                        color:'#111',
                        font:{ size:9, weight:'' },
                        formatter: v => v + '%'
                    }
                },
                {
                    label:'Sisa Hari',
                    data: sisaHari,
                    backgroundColor:'#d1d1d1b4',
                    barThickness: 10,
                    borderRadius: 1,
                    datalabels:{
                        anchor:'end',
                        align:'right',
                        offset:4,
                        color:'#555',
                        font:{ size:9 },
                        formatter: v => v > 0 ? v + ' hari' : ''
                    }
                }
            ]
        },
        options: {
            indexAxis:'y',
            responsive:true,
            maintainAspectRatio:false,
            plugins:{
                legend:{
                    position:'bottom',
                    labels:{ boxWidth:10, font:{ size:11 } }
                },
                tooltip:{ bodyFont:{ size:11 } },
                datalabels:{ clamp:true }
            },
            scales:{
                x:{
                    max:104,
                    ticks:{ callback:v=>v+'', font:{ size:0 } }
                },
                y:{
                    ticks:{ font:{ size:11 } },
                    grid:{ display:false }
                }
            },
            animation:{ duration:600, easing:'easeOutQuart' }
        },
        plugins:[ChartDataLabels]
    });
}

/* drawPieTeam -> menampilkan persentase capaian per tim (realisasi/target *100) */
const pieTeamEl = document.getElementById('pieTeam');

function getContrastColor(hex){
    if(!hex) return '#fff';
    hex = hex.replace('#','');
    if(hex.length === 3){
        hex = hex.split('').map(ch => ch+ch).join('');
    }
    const r = parseInt(hex.substring(0,2),16);
    const g = parseInt(hex.substring(2,4),16);
    const b = parseInt(hex.substring(4,6),16);
    const lum = (0.299 * r + 0.587 * g + 0.114 * b) / 255;
    return lum > 0.6 ? '#111' : '#fff';
}

async function drawPieTeam(){
    try{
        let params = `month=${month.value}&year=${year.value}&activity=${encodeURIComponent(activity.value)}`;
        let res = await fetch(`dashboard.php?action=pie_team&${params}`);
        let j = await res.json();
        if(!Array.isArray(j)) j = [];

        const labels = [];
        const values = [];
        const bg = [];

        const palette = [
  '#3b82f6',
  '#2563eb',
  '#3b82f6',
  '#2f6fed',
  '#3b82f6',
  '#3b82f6',
  '#2f6fed',
  '#3b82f6',
   '#3b82f6',
  '#1d4ed8',
];


        j.forEach((x, idx)=>{
            const teamName = x.team || '—';
            const targ = Number(x.target_total) || 0;
            const real = Number(x.realisasi_total) || 0;
            const percent = targ > 0 ? (real / targ) * 100 : 0;

            labels.push(teamName);
            values.push(Number(percent.toFixed(0)));
            bg.push(palette[idx % palette.length]);
        });

        if(pieTeam) pieTeam.destroy();
        pieTeamEl.style.width = '100%';

        pieTeam = new Chart(pieTeamEl, {
            type: 'bar',
            data: {
                labels,
                datasets: [{
                    label: 'Capaian (%)',
                    data: values,
                    backgroundColor: bg,
                    borderRadius: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,

                // BAR HORIZONTAL
                indexAxis: 'y',

                plugins: {
                    legend: { display: false },
                    tooltip: {
                        callbacks: {
                            label: ctx => {
                                const idx = ctx.dataIndex;
                                const raw = j[idx] || {};
                                const targ = Number(raw.target_total) || 0;
                                const real = Number(raw.realisasi_total) || 0;
                                const pct = values[idx] || 0;
                                return `Capaian: ${pct}% (Realisasi: ${numberWithSep(real)}, Target: ${numberWithSep(targ)})`;
                            }
                        },
                        bodyFont: { size: 10 }
                    },
                    datalabels: {
                        display: true,

                        // untuk horizontal bar
                        anchor: 'end',
                        align: ctx => {
                            const v = ctx.dataset.data[ctx.dataIndex] ?? 0;
                            // bar kecil → label di luar kanan
                            return v < 6 ? 'end' : 'start';
                        },
                        offset: -2,
                        formatter: v => `${v}%`,
                        font: { size: 11, weight: '700' },
                        color: ctx => {
                            let bgc = ctx.dataset.backgroundColor;
                            if(Array.isArray(bgc)) bgc = bgc[ctx.dataIndex];
                            return getContrastColor(bgc);
                        }
                    }
                },
                scales: {
                    y: {
                        ticks: {
                            font: { size: 11 }
                        },
                        grid: { display: false }
                    },
                    x: {
                        beginAtZero: true,
                        max: 100,
                        ticks: {
                            callback: v => v + '%',
                            font: { size: 11 }
                        }
                    }
                },
                animation: {
                    duration: 700,
                    easing: 'easeOutQuart'
                }
            }
        });
    }catch(e){
        console.error('drawPieTeam error', e);
    }
}



/* animasi refresh */
function animateCharts(){
    try{
        if(chart){ chart.reset(); chart.update(); }
        if(chartPegawai){ chartPegawai.reset(); chartPegawai.update(); }
        if(pieTeam){ pieTeam.reset(); pieTeam.update(); }
    }catch(e){ console.error('animateCharts error', e); }
}

/* kegiatan terdekat & hitung mundur */
function renderNearestActivity(obj){
    const el = document.getElementById('nearestActivity');
    const note = document.getElementById('nearestActivityNote');
    el.innerText = '-'; note.innerText = '';
    if(!obj) return;
    const nama = obj.activity_name || '-';
    const teamName = obj.team || '-';
    const sisa = (obj.sisaHari !== undefined && obj.sisaHari !== null) ? obj.sisaHari : '-';
    el.innerText = `Sisa ${sisa} hari untuk kegiatan ${nama} (Tim: ${teamName})`;
    if(obj.deadline){
        const dObj = new Date(obj.deadline);
        if(!isNaN(dObj.getTime())){
            note.innerText = `Deadline: ${dObj.toLocaleDateString('id-ID', { day:'numeric', month:'long', year:'numeric' })}`;
        }
    }
}

function startCountdown(days, deadlineDate){
    clearInterval(countdownInterval);
    const el = document.getElementById('sisaHari');
    const note = document.getElementById('sisaHariNote');
    if(days===null || typeof days==='undefined') days='-';
    note.innerText = '';

    if(days==='-' || days===0 || days<=0){
        el.innerText='-';
        if(deadlineDate){
            const dObj = new Date(deadlineDate);
            if(!isNaN(dObj.getTime())){
                note.innerText = `Sampai ${dObj.toLocaleDateString('id-ID',{day:'numeric',month:'long',year:'numeric'})}`;
            }
        }
        return;
    }

    let end;
    if(deadlineDate){
        const iso = deadlineDate + 'T23:59:59';
        end = new Date(iso).getTime();
        if(isNaN(end)) end = Date.now() + Number(days) * 86400000;
    } else {
        end = Date.now() + Number(days) * 86400000;
    }

    function update(){
        let diff = end - Date.now();
        if(diff <= 0){ el.innerText = 'Habis'; clearInterval(countdownInterval); return; }
        let d = Math.floor(diff/86400000), h = Math.floor((diff%86400000)/3600000),
            m = Math.floor((diff%3600000)/60000), s = Math.floor((diff%60000)/1000);
        el.innerText = `${d}d ${h}h ${m}m ${s}s`;
        if(deadlineDate){
            const dObj = new Date(deadlineDate);
            if(!isNaN(dObj.getTime())){
                note.innerText = `Sisa ${d} hari sampai ${dObj.toLocaleDateString('id-ID',{day:'numeric',month:'long',year:'numeric'})}`;
            }
        }
    }
    update();
    countdownInterval = setInterval(update, 1000);
}

/* --------------------------
   RENDER PODIUM: peringkat berdasarkan "kecepatan"
   - kecepatan diukur sebagai: (realisasi/target) / hariElapsed
   - hariElapsed = hari sejak awal bulan yang dipilih sampai hari ini (min 1)
   - urut berdasarkan speed desc, tie-breaker: capaian desc, realisasi desc
   -------------------------- */
function renderPodium(rekapPegawai){
    const container = document.getElementById('podium');
    container.innerHTML = '';

    // jika tidak ada data, tampilkan placeholder
    if(!Array.isArray(rekapPegawai) || rekapPegawai.length === 0){
        const visualOrder = [4,2,1,3,5];
        const heights = {1:130,2:120,3:110,4:90,5:70};
        visualOrder.forEach(rank=>{
            const h = heights[rank] || 10;
            const itm = document.createElement('div');
            itm.className = 'podium-item';
            itm.innerHTML = `
                <div class="podium-bar pos${rank}" style="height:${h}px">${rank}</div>
                <div class="podium-name">-</div>
                <div class="podium-percent">-</div>
            `;
            container.appendChild(itm);
        });
        return;
    }

    // --- periode: gunakan month/year yang dipilih, jika kosong -> bulan sekarang
    let selMonth = month.value;
    let selYear = year.value;
    const today = new Date();
    if(!selMonth){
        selMonth = String(today.getMonth() + 1).padStart(2,'0');
    }
    if(!selYear){
        selYear = String(today.getFullYear());
    }
    const mIdx = Math.max(0, parseInt(selMonth,10) - 1);
    const periodStart = new Date(parseInt(selYear,10), mIdx, 1);
    // daysElapsed: minimal 1
    let daysElapsed = Math.floor(( (new Date()).getTime() - periodStart.getTime()) / 86400000) + 1;
    if(daysElapsed < 1) daysElapsed = 1;
    // jumlah hari di bulan
    const daysInMonth = new Date(parseInt(selYear,10), mIdx + 1, 0).getDate();

    // hitung skor per pegawai
    const scored = rekapPegawai.map(emp=>{
        const real = Number(emp.realisasi_total) || 0;
        const targ = Number(emp.target_total) || 0;
        const capaian = targ > 0 ? (real / targ) : 0; // fraction 0..1
        // speed = fraction selesai per hari yang telah berlalu (semakin besar semakin baik)
        const speed = targ > 0 ? (capaian / daysElapsed) : 0;
        return Object.assign({}, emp, { real, targ, capaian, speed });
    });

    // urut berdasarkan speed desc, lalu capaian desc, lalu real desc
    scored.sort((a,b)=>{
        if(b.speed !== a.speed) return b.speed - a.speed;
        if(b.capaian !== a.capaian) return b.capaian - a.capaian;
        return b.real - a.real;
    });

    const top5 = scored.slice(0,5);

    // tata letak podium klasik: kiri->kanan = 2,3,1,4,5 (1 berada di tengah tertinggi)
    const visualOrder = [4,2,1,3,5];
    const heightsBase = {1:160,2:120,3:110,4:90,5:70};

    // buat peta rank->data
    const rankMap = {};
    for(let i=0;i<5;i++){
        rankMap[i+1] = top5[i] || null;
    }

    // render
    for(const rank of visualOrder){
        const data = rankMap[rank];
        const fullName = data ? (data.employee || '-') : '-';
        // ambil nama depan saja (perubahan kecil di sini)
        const shortName = (typeof fullName === 'string' && fullName.trim() !== '') ? fullName.trim().split(/\s+/)[0] : fullName;
        const pct  = data ? Math.round((data.capaian||0) * 100) : null;

        // skala tinggi responsif
        let h = heightsBase[rank] || 70;
        const w = container.clientWidth || window.innerWidth;
        if(w < 480) h = Math.round(h * 0.6);
        else if(w < 720) h = Math.round(h * 0.8);

        const item = document.createElement('div');
        item.className = 'podium-item';

        item.innerHTML = `
            <div class="podium-bar pos${rank}" style="height:${h}px">${rank}</div>
            <div class="podium-name" title="${encodeHTML(fullName)}">${encodeHTML(shortName)}</div>
            ${pct !== null ? `<div class="podium-percent">${pct}%</div>` : '<div class="podium-percent">-</div>'}
        `;
        container.appendChild(item);
    }

    // opsional: legenda kecil / catatan di bawah podium (menampilkan metrik yang digunakan)
    // hapus note sebelumnya jika ada
    const existingNote = document.getElementById('podiumNote');
    if(existingNote) existingNote.remove();
    const note = document.createElement('div');
    note.id = 'podiumNote';
    note.className = 'small-note mt-1';
    note.style.textAlign = 'center';
    note.style.fontSize = '12px';
    note.style.color = '#6b7280';
    note.innerText = `Ranking berdasarkan kecepatan penyelesaian (realisasi/target)`;
    container.parentElement.appendChild(note);
}

/* inisialisasi */
(async function init(){
    await loadTeams();
    await loadActivities();
    await loadDashboard();
})();

/* debounce saat resize */
let resizeTimer = null;
window.addEventListener('resize', ()=>{
    if(resizeTimer) clearTimeout(resizeTimer);
    resizeTimer = setTimeout(()=>{ loadDashboard(); }, 300);
});
</script>
</body>
</html>
