<?php
require_once 'config.php';

function json($data){
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

/* ================= API ================= */
if(isset($_GET['action'])){
    // Tim
    if($_GET['action']=='teams'){
        $r=mysqli_query($conn,"SELECT id,name FROM teams ORDER BY name");
        json(mysqli_fetch_all($r,MYSQLI_ASSOC));
    }

    // Pegawai berdasarkan tim
    if($_GET['action']=='employees'){
        $q=mysqli_prepare($conn,"SELECT DISTINCT e.id, e.name 
            FROM employees e
            JOIN activity_master am ON am.employee_id=e.id
            WHERE am.team_id=? 
            ORDER BY e.name");
        mysqli_stmt_bind_param($q,"i",$_GET['team_id']);
        mysqli_stmt_execute($q);
        json(mysqli_fetch_all(mysqli_stmt_get_result($q),MYSQLI_ASSOC));
    }

    // Kegiatan berdasarkan pegawai + tim
    if($_GET['action']=='activities'){
        $q=mysqli_prepare($conn,"SELECT DISTINCT activity_name 
            FROM activity_master 
            WHERE employee_id=? AND team_id=?");
        mysqli_stmt_bind_param($q,"ii", $_GET['employee_id'], $_GET['team_id']);
        mysqli_stmt_execute($q);
        json(mysqli_fetch_all(mysqli_stmt_get_result($q),MYSQLI_ASSOC));
    }

    // Sub-kegiatan berdasarkan pegawai + tim + kegiatan
    // CATATAN: sekarang menerima filter opsional month & year
    if($_GET['action']=='subactivities'){
        $employee_id = isset($_GET['employee_id']) ? intval($_GET['employee_id']) : 0;
        $team_id = isset($_GET['team_id']) ? intval($_GET['team_id']) : 0;
        $activity_name = isset($_GET['activity_name']) ? mysqli_real_escape_string($conn, $_GET['activity_name']) : '';
        $month = isset($_GET['month']) && $_GET['month'] !== '' ? intval($_GET['month']) : '';
        $year  = isset($_GET['year']) && $_GET['year'] !== '' ? intval($_GET['year']) : '';

        $where = "employee_id={$employee_id} AND team_id={$team_id} AND activity_name='{$activity_name}'";
        if($month !== '') $where .= " AND month='".intval($month)."'";
        if($year !== '')  $where .= " AND YEAR(deadline)=".intval($year);

        $q = "SELECT id, sub_activity_name, target_volume, deadline, month 
              FROM activity_master 
              WHERE {$where}
              ORDER BY sub_activity_name";
        $r = mysqli_query($conn, $q);
        json(mysqli_fetch_all($r, MYSQLI_ASSOC));
    }

    // Target & realisasi akumulatif untuk satu sub-kegiatan
    if($_GET['action']=='progress'){
        $q=mysqli_prepare($conn,"SELECT am.target_volume, IFNULL(SUM(de.volume_today),0) AS realisasi, am.deadline
            FROM activity_master am
            LEFT JOIN daily_entries de ON de.activity_master_id=am.id
            WHERE am.id=? GROUP BY am.id");
        mysqli_stmt_bind_param($q,"i",$_GET['activity_id']);
        mysqli_stmt_execute($q);
        json(mysqli_fetch_assoc(mysqli_stmt_get_result($q)));
    }

    // Simpan entri harian
    if($_GET['action']=='save'){
        $activity_id = (int)$_POST['activity_id'];
        $employee_id = (int)$_POST['employee_id'];
        $volume_today = (int)$_POST['volume_today'];

        if($volume_today <= 0) json(['success'=>false,'error'=>'Volume harus lebih dari 0']);

        // Ambil target kegiatan
        $q0=mysqli_prepare($conn,"SELECT target_volume FROM activity_master WHERE id=?");
        mysqli_stmt_bind_param($q0,"i",$activity_id);
        mysqli_stmt_execute($q0);
        $res0=mysqli_stmt_get_result($q0);
        if(!$row0=mysqli_fetch_assoc($res0)) json(['success'=>false,'error'=>'Kegiatan tidak ditemukan']);
        $target_volume=(int)$row0['target_volume'];

        // Hitung realisasi saat ini (akumulatif semua daily_entries untuk activity_master ini)
        $q1=mysqli_prepare($conn,"SELECT IFNULL(SUM(volume_today),0) AS realisasi FROM daily_entries WHERE activity_master_id=?");
        mysqli_stmt_bind_param($q1,"i",$activity_id);
        mysqli_stmt_execute($q1);
        $res1=mysqli_stmt_get_result($q1);
        $row1=mysqli_fetch_assoc($res1);
        $current_realisasi=(int)$row1['realisasi'];

        // Cek apakah target sudah tercapai
        if($target_volume>0 && $current_realisasi >= $target_volume) json(['success'=>false,'error'=>'Target sudah tercapai.']);

        // Cek jika penambahan melebihi target
        if($target_volume>0 && ($current_realisasi+$volume_today)>$target_volume) json(['success'=>false,'error'=>'Realisasi melebihi target.']);

        // Cek ada baris daily_entries untuk kombinasi activity_master_id + employee_id (tanpa membandingkan tanggal)
        $q=mysqli_prepare($conn,"SELECT id,volume_today FROM daily_entries WHERE activity_master_id=? AND employee_id=?");
        mysqli_stmt_bind_param($q,"ii",$activity_id,$employee_id);
        mysqli_stmt_execute($q);
        $res=mysqli_stmt_get_result($q);

        if($row=mysqli_fetch_assoc($res)){
            // Update baris yang ada: jumlahkan volume_today
            $newVol = (int)$row['volume_today'] + $volume_today;
            $q2=mysqli_prepare($conn,"UPDATE daily_entries SET volume_today=? WHERE id=?");
            mysqli_stmt_bind_param($q2,"ii",$newVol,$row['id']);
            mysqli_stmt_execute($q2);
        } else {
            // Belum ada baris: insert baru dan set entry_date ke tanggal sekarang (tanggal pertama tercatat)
            $q2=mysqli_prepare($conn,"INSERT INTO daily_entries(activity_master_id,employee_id,entry_date,volume_today) VALUES(?,?,CURDATE(),?)");
            mysqli_stmt_bind_param($q2,"iii",$activity_id,$employee_id,$volume_today);
            mysqli_stmt_execute($q2);
        }
        json(['success'=>true]);
    }

    // Tabel rekap data
    if($_GET['action']=='list'){
        $month = isset($_GET['month']) && $_GET['month'] !== '' ? intval($_GET['month']) : '';
        $year  = isset($_GET['year']) && $_GET['year'] !== '' ? intval($_GET['year']) : '';

        $where = "";
        if($month !== '') $where .= " WHERE am.month='".intval($month)."'";
        if($year !== ''){
            if($where === "") $where .= " WHERE YEAR(am.deadline)=".intval($year);
            else $where .= " AND YEAR(am.deadline)=".intval($year);
        }

        $r = mysqli_query($conn,"SELECT de.id, de.entry_date,
            e.name employee, t.name team,
            am.activity_name, am.sub_activity_name, am.month,
            am.target_volume, de.volume_today,
            (SELECT IFNULL(SUM(volume_today),0) FROM daily_entries WHERE activity_master_id=am.id) AS realisasi,
            am.deadline
            FROM daily_entries de
            JOIN activity_master am ON am.id=de.activity_master_id
            JOIN employees e ON e.id=de.employee_id
            JOIN teams t ON t.id=am.team_id
            $where
            ORDER BY de.entry_date DESC
        ");
        json(mysqli_fetch_all($r,MYSQLI_ASSOC));
    }

    // Hapus entri
    if($_GET['action']=='delete'){
        $q=mysqli_prepare($conn,"DELETE FROM daily_entries WHERE id=?");
        mysqli_stmt_bind_param($q,"i",$_POST['id']);
        mysqli_stmt_execute($q);
        json(['success'=>true]);
    }
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
<meta charset="UTF-8">
<title>Input Kegiatan Harian</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
<style>
body { font-size: 13px; background: #f5f5f5; }
.container{
  width:100%;
  max-width: min(3840px, 98vw);
  padding-left:12px;
  padding-right:12px;
}
h4 { margin-bottom: 12px; }
.card { border-radius: 12px; box-shadow: 0 4px 15px rgba(0,0,0,0.08); }
.card-body .form-control, .card-body select { border-radius: 8px; }
.btn { border-radius: 8px; font-weight:500; }
.btn i { margin-right:6px; }
.table { background:#fff; border-radius:1px; overflow:hidden; }
.table th { background:#0d6efd; color:#fff; font-size:13px; }
.table-striped tbody tr:nth-of-type(odd){background:#f9f9f9;}
.table-hover tbody tr:hover{background:#e2f0ff;}
.marked td{background:#fff3cd!important;font-weight:600;color:#856404;}
.toast{min-width:250px;border-radius:10px;}
.progress-red{color:#dc3545;font-weight:600;}
.progress-blue{color:#0d6efd;font-weight:600;}
.progress-green{color:#198754;font-weight:600;}
.pagination .page-item .page-link{border-radius:6px;font-size:12px;padding:4px 10px;}
.pagination .page-item.active .page-link{background:#0d6efd;color:#fff;border-color:#0d6efd;}
.pagination .page-item.disabled .page-link{color:#ccc;}
.small-note{font-size:12px;color:#6b7280;margin-bottom:8px;}
/* penyesuaian responsif */
@media (max-width:767px){
    .row.g-2 .col-md-3 { flex: 0 0 50%; max-width: 50%; }
}
</style>
</head>
<body>
<div class="container mt-4">
<h4><i class="bi bi-journal-check"></i> Input Kegiatan Harian Pegawai</h4>

<div class="card mb-3">
  <div class="card-body">
    <div class="small-note">Pilih periode dulu (kiri atas), lalu pilih tim → pegawai → kegiatan → sub-kegiatan. Form akan menyesuaikan secara bertahap.</div>

    <!-- PERIODE -->
    <div class="row g-2 mb-2 align-items-center">
      <div class="col-md-3">
        <label class="form-label small mb-1">Bulan</label>
        <select id="filterMonth" class="form-control">
            <option value="">Semua Bulan</option>
            <?php
            $bulan = [
                '01'=>'Januari','02'=>'Februari','03'=>'Maret','04'=>'April','05'=>'Mei','06'=>'Juni',
                '07'=>'Juli','08'=>'Agustus','09'=>'September','10'=>'Oktober','11'=>'November','12'=>'Desember'
            ];
            $curMonth = date('m');
            foreach ($bulan as $k => $v) {
                $selected = ($k == $curMonth) ? 'selected' : '';
                echo "<option value='$k' $selected>$v</option>";
            }
            ?>
        </select>
      </div>

      <div class="col-md-3">
        <label class="form-label small mb-1">Tahun</label>
        <select id="filterYear" class="form-control">
            <option value="">Tahun</option>
            <?php
            $curYear = date('Y');
            for ($y = $curYear; $y <= $curYear + 2; $y++){
                $sel = ($y == $curYear) ? 'selected' : '';
                echo "<option value='$y' $sel>$y</option>";
            }
            ?>
        </select>
      </div>

      <div class="col-md-6 text-end">
        <div id="currentFilter" class="fw-semibold text-muted" style="margin-top:6px;">Menampilkan: Semua Bulan</div>
      </div>
    </div>

    <!-- IDENTITAS: TIM -> PEGAWAI -> KEGIATAN -> SUB -->
    <div class="row g-2 mb-2">
      <div class="col-md-3">
        <label class="form-label small mb-1">Tim</label>
        <select id="team" class="form-control"><option value="">Pilih Tim</option></select>
      </div>

      <div class="col-md-3">
        <label class="form-label small mb-1">Pegawai</label>
        <!-- tambahkan autocomplete="off" di sini -->
        <input list="employeeList" id="employeeInput" class="form-control" placeholder="Ketikan atau pilih pegawai..." aria-label="Pegawai" disabled autocomplete="off">
        <datalist id="employeeList"></datalist>
        <input type="hidden" id="employee_id" value="">
      </div>

      <div class="col-md-3">
        <label class="form-label small mb-1">Kegiatan</label>
        <!-- tambahkan autocomplete="off" di sini -->
        <input list="activityList" id="activityInput" class="form-control" placeholder="Ketikan atau pilih kegiatan..." aria-label="Kegiatan" disabled autocomplete="off">
        <datalist id="activityList"></datalist>
      </div>

      <div class="col-md-3">
        <label class="form-label small mb-1">Sub Kegiatan</label>
        <select id="subactivity" class="form-control" disabled><option value="">Pilih Sub Kegiatan</option></select>
      </div>
    </div>

    <!-- INFO & INPUT -->
    <div class="row g-2 align-items-end">
      <div class="col-md-3">
        <label class="form-label small mb-1">Target</label>
        <input id="target" class="form-control" readonly placeholder="—">
      </div>

      <div class="col-md-3">
        <label class="form-label small mb-1">Realisasi Akumulatif</label>
        <input id="realisasi" class="form-control" readonly placeholder="—">
      </div>

      <div class="col-md-3">
        <label class="form-label small mb-1">Progress</label>
        <input id="progress" class="form-control" readonly placeholder="—">
      </div>

      <div class="col-md-3">
        <label class="form-label small mb-1">Volume Hari Ini</label>
        <input id="volume_today" type="number" min="1" step="1" class="form-control" placeholder="Masukkan volume" disabled>
      </div>

      <div class="col-12 text-end mt-2">
        <button id="saveBtn" class="btn btn-success" onclick="saveData()" disabled><i class="bi bi-check2-circle"></i> Simpan</button>
      </div>
    </div>

  </div>
</div>

<!-- REKAP TABLE -->
<div class="card">
  <div class="card-body">
    <h6 class="mb-3">Rekap Kegiatan <span id="rekapTitle" class="text-muted" style="font-weight:600;"></span></h6>
    <div class="table-responsive">
      <table class="table table-bordered table-striped table-hover">
      <thead class="text-center">
      <tr>
      <th>No</th><th>Tanggal</th><th>Pegawai</th><th>Tim</th>
      <th>Kegiatan</th><th>Sub</th><th>Target</th>
      <th>Volume Hari Ini</th><th>Realisasi</th><th>Progres</th><th>Sisa Hari</th><th>Status</th>
      </tr>
      </thead>
      <tbody id="data"></tbody>
      </table>
    </div>

    <nav class="mt-2"><ul id="pagination" class="pagination justify-content-center"></ul></nav>
  </div>
</div>

<!-- Toast -->
<div class="position-fixed bottom-0 end-0 p-3" style="z-index:11">
  <div id="successToast" class="toast align-items-center text-bg-success border-0" role="alert" aria-live="assertive" aria-atomic="true">
    <div class="d-flex">
      <div class="toast-body">Data berhasil disimpan!</div>
      <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
    </div>
  </div>
</div>

</div> <!-- container -->

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
// ================= Alur JS =================
let currentTarget=0,currentDeadline=null,currentRealisasi=0,PAGE_SIZE=20,tableData=[],currentPage=1,pageWindowStart=1;

// Referensi DOM
const team = document.getElementById('team');
const employeeInput = document.getElementById('employeeInput');
const employeeIdField = document.getElementById('employee_id');
const activityInput = document.getElementById('activityInput');
const subactivity = document.getElementById('subactivity');
const target = document.getElementById('target');
const realisasi = document.getElementById('realisasi');
const volume_today = document.getElementById('volume_today');
const progress = document.getElementById('progress');
const filterMonth = document.getElementById('filterMonth');
const filterYear  = document.getElementById('filterYear');
const currentFilter = document.getElementById('currentFilter');
const saveBtn = document.getElementById('saveBtn');
const rekapTitle = document.getElementById('rekapTitle');

// cache & peta
const employeesMap = {}; // name -> id
const activitiesCache = {}; // key team_employee -> [names]

// helper
function monthName(m){
    if(!m && m !== 0) return '';
    const n = parseInt(String(m).padStart(2,'0').slice(0,2),10);
    const arr = ['Januari','Februari','Maret','April','Mei','Juni','Juli','Agustus','September','Oktober','November','Desember'];
    return (n>=1 && n<=12) ? arr[n-1] : '';
}
function encodeHTML(s){ if(!s && s !== 0) return ''; return String(s).replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;').replace(/"/g,'&quot;').replace(/'/g,'&#39;'); }

// perbarui state UI (aktif/non-aktif field) berdasarkan pemilihan saat ini
function updateUIState(){
    employeeInput.disabled = !team.value;
    const hasEmpId = !!employeeIdField.value;
    activityInput.disabled = !(team.value && hasEmpId);
    subactivity.disabled = !(team.value && hasEmpId && activityInput.value);

    const sel = subactivity.selectedOptions[0];
    if(sel && sel.value){
        if(currentTarget && currentRealisasi >= currentTarget){
            volume_today.disabled = true;
            volume_today.placeholder = 'Target tercapai';
            saveBtn.disabled = true;
        } else {
            volume_today.disabled = false;
            volume_today.placeholder = 'Masukkan volume';
            saveBtn.disabled = false;
        }
    } else {
        volume_today.disabled = true;
        saveBtn.disabled = true;
    }
}

/* muat daftar tim */
async function loadTeams(){
    try{
        let r = await fetch('inputharian.php?action=teams');
        let j = await r.json();
        team.innerHTML = '<option value="">Pilih Tim</option>';
        j.forEach(x=> team.innerHTML += `<option value="${x.id}">${encodeHTML(x.name)}</option>`);
    }catch(e){ console.error('loadTeams', e); }
}

/* Bantu: bangun ulang datalist dari array string */
function rebuildDatalist(listId, items){
    const listEl = document.getElementById(listId);
    if(!listEl) return;
    listEl.innerHTML = '';
    items.forEach(v=>{
        const opt = document.createElement('option');
        opt.value = v;
        listEl.appendChild(opt);
    });
}

/* Filter nilai datalist berdasarkan substring (tanpa memperhatikan huruf besar/kecil) */
function filterDatalist(listId, allItems, query){
    const q = (query||'').toLowerCase().trim();
    if(!q) {
        rebuildDatalist(listId, allItems);
        return;
    }
    const filtered = allItems.filter(n => n.toLowerCase().includes(q));
    rebuildDatalist(listId, filtered);
}

// saat tim berubah -> muat pegawai dan reset field turunan
team.onchange = async function(){
    employeeInput.value = '';
    employeeIdField.value = '';
    activityInput.value = '';
    subactivity.innerHTML = '<option value="">Pilih Sub Kegiatan</option>';
    target.value=''; realisasi.value=''; progress.value=''; volume_today.value='';
    currentTarget=0; currentRealisasi=0; currentDeadline=null;

    // bersihkan datalist pegawai & map
    document.getElementById('employeeList').innerHTML = '';
    for(const k in employeesMap) delete employeesMap[k];

    if(!team.value){
        updateUIState();
        await loadTable();
        return;
    }

    try{
        let r = await fetch(`inputharian.php?action=employees&team_id=${team.value}`);
        let j = await r.json();
        const names = [];
        j.forEach(x=>{
            names.push(x.name);
            employeesMap[x.name] = x.id;
        });
        rebuildDatalist('employeeList', names);
    }catch(e){ console.error('loadEmployees', e); }

    document.getElementById('activityList').innerHTML = '';
    updateUIState();
    await loadTable();
};

// input pada employeeInput: filter datalist & set employee_id hanya ketika ada kecocokan tepat
employeeInput.addEventListener('input', async function(){
    const name = this.value.trim();
    const allNames = Object.keys(employeesMap);
    // filter datalist tampilan
    filterDatalist('employeeList', allNames, name);

    // bersihkan turunan sampai pegawai valid dipilih
    activityInput.value = '';
    document.getElementById('activityList').innerHTML = '';
    subactivity.innerHTML = '<option value="">Pilih Sub Kegiatan</option>';
    target.value=''; realisasi.value=''; progress.value=''; volume_today.value='';
    currentTarget=0; currentRealisasi=0; currentDeadline=null;
    employeeIdField.value = '';

    if(!name || !team.value) {
        updateUIState();
        await loadTable();
        return;
    }

    // kecocokan tepat -> set id dan muat kegiatan (case-insensitive)
    const exactKey = Object.keys(employeesMap).find(k => k.toLowerCase() === name.toLowerCase());
    if(exactKey){
        // normalisasi tampilan nama sesuai database
        employeeInput.value = exactKey;
        employeeIdField.value = employeesMap[exactKey];
        await loadActivitiesFor(team.value, employeeIdField.value);
    }

    updateUIState();
    await loadTable();
});

/* PENTING: saat blur, jika input TIDAK cocok dengan data nyata -> kosongkan.
   Ini mencegah teks ketikan yang bukan dari DB tetap terlihat. */
employeeInput.addEventListener('blur', function(){
    const name = this.value.trim();
    if(!name){
        employeeIdField.value = '';
        this.value = '';
        return;
    }
    const exactKey = Object.keys(employeesMap).find(k => k.toLowerCase() === name.toLowerCase());
    if(!exactKey){
        // hapus nilai yang tidak cocok
        employeeIdField.value = '';
        this.value = '';
    } else {
        // normalisasi kapitalisasi
        this.value = exactKey;
        employeeIdField.value = employeesMap[exactKey];
    }
});

/* muat kegiatan untuk kombinasi tim+pegawai, simpan cache dan isi datalist */
async function loadActivitiesFor(teamId, employeeId){
    if(!teamId || !employeeId) return;
    const key = `${teamId}_${employeeId}`;
    try{
        let r = await fetch(`inputharian.php?action=activities&employee_id=${employeeId}&team_id=${teamId}`);
        let rows = await r.json();
        const names = rows.map(r=> (r.activity_name || (Array.isArray(r) ? r[0] : r)).trim()).filter(s=>s!=='');

        activitiesCache[key] = names;
        rebuildDatalist('activityList', names);
    }catch(e){ console.error('loadActivitiesFor', e); document.getElementById('activityList').innerHTML = ''; activitiesCache[key]=[]; }
}

/* input pada activityInput:
   - filter datalist untuk menampilkan nama kegiatan yang cocok
   - hanya ambil sub-kegiatan saat nama persis cocok dengan data (case-insensitive)
*/
activityInput.addEventListener('input', async function(){
    const activityName = this.value.trim();
    const actListEl = document.getElementById('activityList');
    subactivity.innerHTML = '<option value="">Pilih Sub Kegiatan</option>';
    target.value=''; realisasi.value=''; progress.value=''; volume_today.value='';
    currentTarget=0; currentRealisasi=0; currentDeadline=null;

    if(!team.value || !employeeIdField.value){
        updateUIState();
        await loadTable();
        return;
    }

    const key = `${team.value}_${employeeIdField.value}`;
    // jika cache belum ada, ambil kegiatan dulu
    if(!activitiesCache[key]) {
        await loadActivitiesFor(team.value, employeeIdField.value);
    }

    const allNames = activitiesCache[key] || [];
    // filter datalist berdasarkan input saat ini (cocok substring)
    filterDatalist('activityList', allNames, activityName);

    if(!activityName){
        updateUIState();
        await loadTable();
        return;
    }

    // hanya ambil sub-kegiatan bila activityName cocok persis dengan salah satu nama kegiatan (case-insensitive)
    const exactName = allNames.find(n => n.toLowerCase() === activityName.toLowerCase());
    if(!exactName){
        // jangan ambil sub untuk input parsial; tetap bersihkan select subs
        updateUIState();
        await loadTable();
        return;
    }

    // normalisasi input ke nama yang benar (koreksi kapitalisasi)
    activityInput.value = exactName;

    // kecocokan tepat -> ambil subkegiatan
    try{
        const m = encodeURIComponent(filterMonth.value);
        const y = encodeURIComponent(filterYear.value);
        const url = `inputharian.php?action=subactivities&employee_id=${employeeIdField.value}&team_id=${team.value}&activity_name=${encodeURIComponent(exactName)}&month=${m}&year=${y}`;
        let r = await fetch(url);
        let rows = await r.json();
        rows.forEach(x=>{
            const id = x.id;
            const sub = x.sub_activity_name || '-';
            const t = x.target_volume || 0;
            const dl = x.deadline || '';
            const opt = document.createElement('option');
            opt.value = id;
            opt.dataset.target = t;
            opt.dataset.deadline = dl;
            opt.textContent = sub;
            subactivity.appendChild(opt);
        });
    }catch(e){ console.error('loadSubactivities', e); }
    updateUIState();
    await loadTable();
});

/* saat blur pada activityInput: kosongkan nilai yang tidak cocok sehingga hanya data dari DB yang tersisa */
activityInput.addEventListener('blur', function(){
    const v = this.value.trim();
    if(!v){
        this.value = '';
        // kosongkan subs
        subactivity.innerHTML = '<option value="">Pilih Sub Kegiatan</option>';
        updateUIState();
        return;
    }
    const key = `${team.value}_${employeeIdField.value}`;
    const allNames = activitiesCache[key] || [];
    const exactName = allNames.find(n => n.toLowerCase() === v.toLowerCase());
    if(!exactName){
        // bukan data nyata -> kosongkan
        this.value = '';
        subactivity.innerHTML = '<option value="">Pilih Sub Kegiatan</option>';
    } else {
        // normalisasi kapitalisasi
        this.value = exactName;
    }
});

/* saat memilih subactivity -> muat progress (target & realisasi) */
subactivity.onchange = async function(){
    const sel = subactivity.selectedOptions[0];
    currentTarget = 0; currentRealisasi = 0; currentDeadline = null;
    target.value=''; realisasi.value=''; progress.value=''; volume_today.value='';
    if(!sel || !sel.value){ updateUIState(); return; }
    currentTarget = parseInt(sel.dataset.target || 0);
    currentDeadline = sel.dataset.deadline || null;
    try{
        let res = await fetch(`inputharian.php?action=progress&activity_id=${sel.value}`);
        let data = await res.json();
        target.value = currentTarget || 0;
        realisasi.value = data.realisasi || 0;
        currentRealisasi = parseInt(data.realisasi || 0);
        let persen = currentTarget ? Math.round((currentRealisasi/currentTarget)*100) : 0;
        progress.value = persen + '%';
    }catch(e){ console.error('progress', e); }
    updateUIState();
};

/* perubahan filter -> muat ulang subactivities jika activity cocok, dan muat tabel */
filterMonth.onchange = async function(){
    updateFilterLabels();
    if(activityInput.value && employeeIdField.value && team.value){
        // trig input untuk mengecek kecocokan tepat dan ambil subs bila perlu
        activityInput.dispatchEvent(new Event('input'));
    }
    await loadTable();
};
filterYear.onchange = async function(){
    updateFilterLabels();
    if(activityInput.value && employeeIdField.value && team.value){
        activityInput.dispatchEvent(new Event('input'));
    }
    await loadTable();
};

function updateFilterLabels(){
    const m = filterMonth.value;
    const y = filterYear.value;
    let label = '';
    if(m && y) label = `${monthName(m)} ${y}`;
    else if(m) label = `${monthName(m)}`;
    else if(y) label = `${y}`;
    else label = 'Semua Bulan';
    currentFilter.innerText = `Menampilkan: ${label}`;
    rekapTitle.innerText = (label === 'Semua Bulan') ? '' : `— ${label}`;
}

/* preview progress saat mengetik volume */
volume_today.oninput = function(){
    let v = parseInt(volume_today.value || 0);
    if(!subactivity.selectedOptions[0]) return;
    if(v <= 0){ progress.value = ''; saveBtn.disabled = true; return; }
    let total = currentRealisasi + v;
    if(currentTarget && total > currentTarget){
        progress.value = '';
        saveBtn.disabled = true;
        return;
    }
    let persen = currentTarget ? Math.round((total/currentTarget)*100) : 0;
    progress.value = persen + '%';
    saveBtn.disabled = false;
};

/* simpan entri */
async function saveData(){
    if(volume_today.disabled) return alert('Target sudah tercapai');
    if(!subactivity.value || !volume_today.value) return alert('Lengkapi data');
    let v = parseInt(volume_today.value || 0);
    if(v <= 0) return alert('Volume harus lebih dari 0');
    if(currentTarget && (currentRealisasi + v) > currentTarget) { alert('⚠️ Tidak bisa disimpan!'); return; }

    if(!employeeIdField.value) return alert('Pilih pegawai yang valid.');

    let fd = new FormData();
    fd.append('activity_id', subactivity.value);
    fd.append('employee_id', employeeIdField.value);
    fd.append('volume_today', v);

    try{
        let res = await fetch('inputharian.php?action=save', { method: 'POST', body: fd });
        let j = await res.json();
        if(!j.success){ alert('Gagal menyimpan: '+(j.error||'Terjadi kesalahan')); return; }
        let toast = new bootstrap.Toast(document.getElementById('successToast'), { delay: 2000 });
        toast.show();
        volume_today.value = ''; progress.value = ''; 
        if(subactivity.value) await subactivity.onchange();
        await loadTable();
    }catch(e){ console.error('saveData', e); alert('Terjadi kesalahan saat menyimpan'); }
}

/* TABEL: rendering sama seperti sebelumnya */
async function loadTable(){
    updateFilterLabels();
    const m = encodeURIComponent(filterMonth.value);
    const y = encodeURIComponent(filterYear.value);
    try{
        let r = await fetch(`inputharian.php?action=list&month=${m}&year=${y}`);
        tableData = await r.json();
    }catch(e){ console.error('loadTable', e); tableData = []; }
    currentPage = 1; pageWindowStart = 1;
    renderTablePage(currentPage);
    renderPagination();
}

function renderTablePage(page){
    currentPage = page;
    const totalPages = Math.max(1, Math.ceil(tableData.length / PAGE_SIZE));
    if(currentPage < 1) currentPage = 1;
    if(currentPage > totalPages) currentPage = totalPages;
    const start = (currentPage - 1) * PAGE_SIZE;
    const items = tableData.slice(start, start + PAGE_SIZE);
    let h = '', i = start + 1;
    items.forEach(x=>{
        let p = x.target_volume ? Math.round((x.realisasi / x.target_volume) * 100) : 0;
        let progressClass = 'progress-red';
        if(p >= 100) progressClass = 'progress-green';
        else if(p >= 51) progressClass = 'progress-blue';
        let sisaHari = x.deadline ? Math.max(0, Math.ceil((new Date(x.deadline) - new Date()) / 86400000)) : '-';
        const subLabel = x.sub_activity_name ? x.sub_activity_name : '-';
        let statusBadge = p >= 100 ? '<span class="badge bg-success px-2 py-1" style="font-size:9px;">Selesai</span>' : '<span class="badge bg-danger px-2 py-1" style="font-size:9px;">Proses</span>';
        h += `<tr>
            <td class="text-center">${i++}</td>
            <td class="text-center">${encodeHTML(x.entry_date)}</td>
            <td>${encodeHTML(x.employee)}</td>
            <td>${encodeHTML(x.team)}</td>
            <td>${encodeHTML(x.activity_name)}</td>
            <td>${encodeHTML(subLabel)}</td>
            <td class="text-end">${encodeHTML(x.target_volume)}</td>
            <td class="text-end">${encodeHTML(x.volume_today)}</td>
            <td class="text-end">${encodeHTML(x.realisasi)}</td>
            <td class="${progressClass} text-center">${p}%</td>
            <td class="text-center">Sisa ${sisaHari} Hari</td>
            <td class="text-center">${statusBadge}</td>
        </tr>`;
    });
    document.getElementById('data').innerHTML = h;
    applyHighlight();
}

function applyHighlight(){
    const selTeam = team.value ? team.selectedOptions[0].text : '';
    const selEmployee = employeeInput.value || '';
    const selActivity = activityInput.value || '';
    const selSub = subactivity.selectedOptions[0] ? subactivity.selectedOptions[0].text : '';

    document.querySelectorAll('#data tr').forEach(tr=>{
        tr.classList.remove('marked');
        if(!selTeam && !selEmployee && !selActivity && !selSub) return;
        let match = true;
        if(selTeam && tr.children[3].innerText !== selTeam) match = false;
        if(selEmployee && tr.children[2].innerText !== selEmployee) match = false;
        if(selActivity && tr.children[4].innerText !== selActivity) match = false;
        if(selSub && tr.children[5].innerText !== selSub) match = false;
        if(match) tr.classList.add('marked');
    });
}

/* pagination (sama seperti sebelumnya) */
function renderPagination(){
    const totalPages = Math.max(1, Math.ceil(tableData.length / PAGE_SIZE));
    const ul = document.getElementById('pagination'); ul.innerHTML = '';
    const prevLi = document.createElement('li'); prevLi.className = 'page-item' + (currentPage === 1 ? ' disabled' : '');
    prevLi.innerHTML = '<a class="page-link" href="#">&laquo;</a>';
    prevLi.onclick = e => { e.preventDefault(); if(currentPage > 1) renderTablePage(currentPage - 1); };
    ul.appendChild(prevLi);

    if(pageWindowStart > 1){
        const li = document.createElement('li'); li.className='page-item';
        li.innerHTML = '<a class="page-link" href="#">...</a>';
        li.onclick = e => { e.preventDefault(); pageWindowStart = Math.max(1, pageWindowStart - 3); renderPagination(); };
        ul.appendChild(li);
    }

    for(let p = pageWindowStart; p <= Math.min(pageWindowStart + 2, totalPages); p++){
        const li = document.createElement('li');
        li.className = 'page-item' + (p === currentPage ? ' active' : '');
        li.innerHTML = `<a class="page-link" href="#">${p}</a>`;
        li.onclick = e => { e.preventDefault(); renderTablePage(p); };
        ul.appendChild(li);
    }

    if(pageWindowStart + 2 < totalPages){
        const li = document.createElement('li'); li.className='page-item';
        li.innerHTML = '<a class="page-link" href="#">...</a>';
        li.onclick = e => { e.preventDefault(); pageWindowStart = pageWindowStart + 3; renderPagination(); };
        ul.appendChild(li);
    }

    const nextLi = document.createElement('li'); nextLi.className = 'page-item' + (currentPage === totalPages ? ' disabled' : '');
    nextLi.innerHTML = '<a class="page-link" href="#">&raquo;</a>';
    nextLi.onclick = e => { e.preventDefault(); if(currentPage < totalPages) renderTablePage(currentPage + 1); };
    ul.appendChild(nextLi);
}

/* pembantu untuk tombol Simpan */
function checkSaveEnable(){
    if(subactivity.value && !volume_today.disabled && volume_today.value && Number(volume_today.value) > 0){
        saveBtn.disabled = false;
    } else {
        saveBtn.disabled = true;
    }
}

/* pasang event listener tambahan */
[filterMonth.id, filterYear.id].forEach(id=>{
    const el = document.getElementById(id);
    if(el) el.addEventListener('change', ()=>{ updateUIState(); checkSaveEnable(); });
});
volume_today.addEventListener('input', checkSaveEnable);
employeeInput.addEventListener('input', ()=>{ updateUIState(); checkSaveEnable(); });
activityInput.addEventListener('input', ()=>{ updateUIState(); checkSaveEnable(); });
subactivity.addEventListener('change', ()=>{ updateUIState(); checkSaveEnable(); });

/* inisialisasi */
(async function init(){
    await loadTeams();
    updateFilterLabels();
    updateUIState();
    await loadTable();
})();

</script>
</body>
</html>
