<?php
require_once 'config.php';

function json($data){
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

/**
 * Helper: bind params ke prepared statement kompatibel dengan berbagai versi PHP
 * $stmt : mysqli_stmt
 * $types: string, mis. "is"
 * $params: array of values
 */
function stmt_bind_params($stmt, $types, $params){
    if(empty($params)) return;
    // prepare array of references
    $refs = [];
    $refs[] = $types;
    foreach($params as $k => $v){
        // make sure each param is a variable and passed by reference
        $refs[] = &$params[$k];
    }
    call_user_func_array([$stmt, 'bind_param'], $refs);
}

/* ================= API ================= */
if(isset($_GET['action'])){
    if($_GET['action']=='employees'){
        $r = mysqli_query($conn,"SELECT DISTINCT e.id, e.name 
            FROM employees e 
            JOIN activity_master am ON am.employee_id=e.id 
            ORDER BY e.name");
        json(mysqli_fetch_all($r,MYSQLI_ASSOC));
    }

    if($_GET['action']=='rekap'){
        $employee_id = $_GET['employee_id'] ?? null;
        $month = $_GET['month'] ?? null;
        $search = $_GET['search'] ?? null;

        $sql = "SELECT e.name AS employee, t.name AS team, am.activity_name, am.sub_activity_name, am.target_volume,
                IFNULL(SUM(de.volume_today),0) AS realisasi,
                -- lebih aman: hindari pembagian dengan 0
                CASE WHEN am.target_volume IS NOT NULL AND am.target_volume>0 
                     THEN ROUND(IFNULL(SUM(de.volume_today),0)/am.target_volume*100,0) 
                     ELSE 0 END AS capaian,
                DATE_FORMAT(am.deadline,'%Y-%m-%d') AS deadline
                FROM activity_master am
                JOIN employees e ON e.id=am.employee_id
                JOIN teams t ON t.id=am.team_id
                LEFT JOIN daily_entries de ON de.activity_master_id=am.id
                WHERE 1 ";

        $params = [];
        $types = "";

        if($employee_id){
            $sql .= " AND am.employee_id=? ";
            $types .= "i";
            $params[] = (int)$employee_id;
        }

        if($month){
            // filter berdasarkan bulan di activity_master (seperti pada target kegiatan)
            $sql .= " AND am.month=? ";
            $types .= "i";
            $params[] = (int)$month;
        }

        if($search){
            $sql .= " AND am.activity_name LIKE ? ";
            $types .= "s";
            $params[] = "%$search%";
        }

        $sql .= " GROUP BY am.id ORDER BY e.name, t.name, am.activity_name";

        $stmt = mysqli_prepare($conn,$sql);
        if($stmt === false){
            json(['error'=>'Prepare failed','mysql_error'=>mysqli_error($conn)]);
        }

        if($params){
            stmt_bind_params($stmt,$types,$params);
        }

        mysqli_stmt_execute($stmt);
        $res = mysqli_stmt_get_result($stmt);
        json(mysqli_fetch_all($res,MYSQLI_ASSOC));
    }
}
?>
<!doctype html>
<html lang="id">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>Rekap Kegiatan Pegawai</title>

<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">

<style>
body{font-size:13px;background:#f1f3f6}
.container{
  width:100%;
  max-width: min(3840px, 98vw);
  padding-left:12px;
  padding-right:12px;
}

/* HEADER */
.header-bar{
  display:flex;
  justify-content:space-between;
  align-items:center;
  margin-bottom:8px;
}
.page-title{
  display:flex;
  align-items:center;
  gap:8px;
  font-weight:600;
  font-size:1.2rem;
}

/* FILTER MODERN */
.header-right{
  display:flex;
  gap:8px;
}
.filter-group{
  position:relative;
}
.filter-group input,
.filter-group select{
  height:34px;
  font-size:13px;
  padding:7px 20px 7px 20px; /* space for icon */
  border-radius:8px;
  border:1px solid #ddd;
  background:#fff;
}
.filter-group .icon{
  position:absolute;
  left:8px;
  top:50%;
  transform:translateY(-50%);
  color:#999;
  pointer-events:none;
}

/* CARD + TABLE */
.card{
  border:none;
  border-radius:12px;
  box-shadow:0 4px 14px rgba(0,0,0,.06);
}
/* ubah ukuran font untuk seluruh tabel menjadi 10px */
.table, .table th, .table td, #rekap-table, #rekap-table th, #rekap-table td {
  font-size:10px !important;
}
.table th{
  /* sebelumnya 12px; sekarang diseragamkan ke 10px */
  text-transform:uppercase;
}
.table tbody tr:hover{
  background:#f8f9fc;
}

/* CAPAIAN BADGE */
.cap{
  padding:2px 7px;
  border-radius:6px;
  font-size:11px;
  font-weight:600;
  display:inline-block;
}
.ok{background:#d1e7dd;color:#0f5132}
.mid{background:#fff3cd;color:#664d03}
.low{background:#f8d7da;color:#842029}

/* PAGINATION */
.pagination{
  gap:4px;
  margin-top:10px;
}
.pagination button{
  border:none;
  background:#fff;
  border-radius:6px;
  padding:4px 10px;
  font-size:12px;
  box-shadow:0 1px 4px rgba(0,0,0,.1);
}
.pagination button.active{
  background:#0d6efd;
  color:#fff;
}

/* Print adjustments (narrower margins, suitable font sizes) */
@media print {
  body { background: #fff; }
  .no-print { display:none !important; }
  table { page-break-inside: auto; }
  tr { page-break-inside: avoid; page-break-after: auto; }
  /* pastikan print juga pakai font 10px untuk tabel */
  .table, .table th, .table td, #rekap-table, #rekap-table th, #rekap-table td {
    font-size:10px !important;
  }
}
</style>
</head>

<body>
<div class="container mt-1">

<!-- HEADER -->
<div class="header-bar">
  <div class="page-title">
    <i class="bi bi-file-earmark-text text-primary fs-4"></i>
    Rekap Kegiatan Pegawai
  </div>

  <div class="header-right">
    <!-- Pegawai -->
    <div class="filter-group">
      <i class="bi bi-person icon"></i>
      <select id="employee"></select>
    </div>

    <!-- Bulan -->
    <div class="filter-group">
      <i class="bi bi-calendar icon"></i>
      <select id="month">
        <option value="">Semua Bulan</option>
        <?php
        $bulan=["","Januari","Februari","Maret","April","Mei","Juni","Juli","Agustus","September","Oktober","November","Desember"];
        for($i=1;$i<=12;$i++) echo "<option value='$i'>{$bulan[$i]}</option>";
        ?>
      </select>
    </div>

    <!-- Search -->
    <div class="filter-group">
      <i class="bi bi-search icon"></i>
      <input type="text" id="search" placeholder="Cari nama kegiatan...">
    </div>

    <!-- TOMBOL CETAK (DITAMBAHKAN) -->
    <div class="d-flex align-items-center">
      <button class="btn btn-outline-primary btn-sm no-print" onclick="cetakData()">
        <i class="bi bi-printer"></i> Cetak
      </button>
    </div>
  </div>
</div>

<!-- TABLE -->
<div class="table-responsive">
<table class="table table-bordered table-striped align-middle mb-0" id="rekap-table">
<thead class="table-dark text-center">
<tr>
<th>No</th>
<th>Pegawai</th>
<th>Tim</th>
<th>Kegiatan</th>
<th>Sub</th>
<th>Target</th>
<th>Realisasi</th>
<th>%</th>
<th>Deadline</th>
</tr>
</thead>
<tbody id="data"></tbody>
</table>
</div>

<div class="pagination d-flex justify-content-center no-print" id="pagination"></div>

</div>

<script>
const employee = document.getElementById('employee');
const month = document.getElementById('month');
const search = document.getElementById('search');
const dataEl = document.getElementById('data');
const paginationEl = document.getElementById('pagination');

let allData=[], currentPage=1;
const perPage=10;

/* load employees */
async function loadEmployees(){
 let r = await fetch('rekapkegiatan.php?action=employees');
 let j = await r.json();
 employee.innerHTML='<option value="">Semua Pegawai</option>';
 j.forEach(x=>employee.innerHTML+=`<option value="${x.id}">${x.name}</option>`);
}

/* load data */
async function loadTable(){
 let url = `rekapkegiatan.php?action=rekap`;
 if(employee.value) url+=`&employee_id=${employee.value}`;
 if(month.value) url+=`&month=${month.value}`;
 if(search.value.trim()) url+=`&search=${encodeURIComponent(search.value.trim())}`;

 let r = await fetch(url);
 allData = await r.json();
 currentPage=1;
 render();
}

/* render table */
function render(){
 let start=(currentPage-1)*perPage;
 let slice=allData.slice(start,start+perPage);
 let h='',i=start+1;

 slice.forEach(x=>{
   // pastikan capaian numeric
   let capa = Number(x.capaian);
   let cls = capa>=80?'ok':capa>=50?'mid':'low';
   h+=`<tr>
    <td class="text-center">${i++}</td>
    <td>${x.employee}</td>
    <td>${x.team}</td>
    <td>${x.activity_name}</td>
    <td>${x.sub_activity_name||'-'}</td>
    <td class="text-end">${x.target_volume}</td>
    <td class="text-end">${x.realisasi}</td>
    <td class="text-center"><span class="cap ${cls}">${x.capaian}%</span></td>
    <td class="text-center">${x.deadline||'-'}</td>
   </tr>`;
 });
 dataEl.innerHTML=h||'<tr><td colspan="9" class="text-center">Tidak ada data</td></tr>';
 renderPagination();
}

/* pagination: max 3 angka + panah */
function renderPagination(){
 let totalPages=Math.ceil(allData.length/perPage);
 if(totalPages<=1){paginationEl.innerHTML='';return;}

 let html='';
 html+=`<button ${currentPage==1?'disabled':''} onclick="goto(${currentPage-1})">&laquo;</button>`;

 let start=Math.max(1,currentPage-1);
 let end=Math.min(totalPages,start+2);
 if(end-start<2) start=Math.max(1,end-2);

 for(let i=start;i<=end;i++){
   html+=`<button class="${i==currentPage?'active':''}" onclick="goto(${i})">${i}</button>`;
 }

 html+=`<button ${currentPage==totalPages?'disabled':''} onclick="goto(${currentPage+1})">&raquo;</button>`;
 paginationEl.innerHTML=html;
}

function goto(p){currentPage=p;render();}

employee.addEventListener('change',loadTable);
month.addEventListener('change',loadTable);
search.addEventListener('input',loadTable);

loadEmployees();
loadTable();

/* -------------------------
   Fungsi Cetak: langsung mencetak isi tabel sesuai filter saat ini
   - menggunakan hidden iframe untuk menulis HTML print-friendly
   - menyalin header, filter saat ini, dan isi tabel yang terlihat (dataEl)
   ------------------------- */
function cetakData(){
  // Ambil teks filter yang sedang dipilih
  const empText = employee.options[employee.selectedIndex]?.text || 'Semua Pegawai';
  const monthText = month.options[month.selectedIndex]?.text || 'Semua Bulan';
  const searchText = search.value.trim() || '-';

  // Ambil markup table (thead + tbody yang sedang terlihat/paging)
  const table = document.getElementById('rekap-table').cloneNode(true);
  const newTbody = document.createElement('tbody');
  if(allData.length===0){
    newTbody.innerHTML = '<tr><td colspan="9" class="text-center">Tidak ada data</td></tr>';
  } else {
    let no=1;
    allData.forEach(x=>{
      const capa = Number(x.capaian);
      const cls = capa>=80?'ok':capa>=50?'mid':'low';
      const tr = document.createElement('tr');
      tr.innerHTML = `<td class="text-center">${no++}</td>
        <td>${x.employee}</td>
        <td>${x.team}</td>
        <td>${x.activity_name}</td>
        <td>${x.sub_activity_name||'-'}</td>
        <td class="text-end">${x.target_volume}</td>
        <td class="text-end">${x.realisasi}</td>
        <td class="text-center"><span class="cap ${cls}">${x.capaian}%</span></td>
        <td class="text-center">${x.deadline||'-'}</td>`;
      newTbody.appendChild(tr);
    });
  }
  // replace tbody in cloned table
  const oldTbody = table.querySelector('tbody');
  if(oldTbody) oldTbody.parentNode.replaceChild(newTbody, oldTbody);

  // Siapkan HTML untuk dicetak: sertakan bootstrap dan style internal supaya tampilan konsisten
  const styles = document.querySelector('style').outerHTML; // ambil style halaman saat ini
  const bootstrapHref = 'https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css';
  const iconsHref = 'https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css';
  const title = document.title || 'Rekap Kegiatan Pegawai';

  const headerHtml = `
    <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:12px;">
      <div>
        <h4 style="margin:0">${title}</h4>
        <div style="font-size:13px;color:#555;margin-top:6px;">
          <strong>Pegawai:</strong> ${escapeHtml(empText)} &nbsp;|&nbsp;
          <strong>Bulan:</strong> ${escapeHtml(monthText)} &nbsp;|&nbsp;
          <strong>Filter:</strong> ${escapeHtml(searchText)}
        </div>
      </div>
      <div style="text-align:right;font-size:12px;color:#666;">
        Dicetak: ${new Date().toLocaleString()}
      </div>
    </div>
  `;

  const html = `<!doctype html>
  <html>
  <head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <title>${escapeHtml(title)}</title>
    <link rel="stylesheet" href="${bootstrapHref}">
    <link rel="stylesheet" href="${iconsHref}">
    ${styles}
    <style>
      /* Print-specific tweaks */
      body { padding: 18px; color:#222; }
      table { width:100%; border-collapse:collapse; font-size:10px; } /* DISINI: 10px */
      .cap { font-weight:600; padding:2px 6px; border-radius:6px; display:inline-block; }
      .ok{background:#d1e7dd;color:#0f5132}
      .mid{background:#fff3cd;color:#664d03}
      .low{background:#f8d7da;color:#842029}
      @media print {
        @page { margin: 12mm; }
        .no-print { display:none !important; }
      }
      /* Pastikan header/table pada halaman cetak juga mengikuti ukuran font 10px */
      .table, .table th, .table td, #rekap-table, #rekap-table th, #rekap-table td {
        font-size:10px !important;
      }
    </style>
  </head>
  <body>
    ${headerHtml}
    ${table.outerHTML}
  </body>
  </html>`;

  // Buat iframe tersembunyi untuk mencetak
  let iframe = document.getElementById('print-iframe-temp');
  if(iframe) document.body.removeChild(iframe);

  iframe = document.createElement('iframe');
  iframe.style.width = '0';
  iframe.style.height = '0';
  iframe.style.border = '0';
  iframe.id = 'print-iframe-temp';
  iframe.className = 'no-print';
  document.body.appendChild(iframe);

  const doc = iframe.contentWindow || iframe.contentDocument;
  if(iframe.contentDocument) {
    iframe.contentDocument.open();
    iframe.contentDocument.write(html);
    iframe.contentDocument.close();
  } else if(iframe.contentWindow) {
    iframe.contentWindow.document.open();
    iframe.contentWindow.document.write(html);
    iframe.contentWindow.document.close();
  }

  // Setelah iframe siap, panggil print
  iframe.onload = function(){
    try{
      iframe.contentWindow.focus();
      // beberapa browser butuh sedikit delay agar resources (CSS) ter-render
      setTimeout(function(){
        iframe.contentWindow.print();
        // Hapus iframe setelah print (beri sedikit waktu agar print dialog muncul)
        setTimeout(function(){ 
          if(iframe && iframe.parentNode) iframe.parentNode.removeChild(iframe);
        }, 500);
      }, 200);
    }catch(e){
      console.error('Print error:', e);
      // fallback: buka window baru jika iframe gagal
      const w = window.open('', '_blank');
      if(w){
        w.document.write(html);
        w.document.close();
        w.focus();
        w.print();
        w.close();
      } else {
        alert('Gagal memulai proses cetak. Periksa popup blocker atau coba cara lain.');
      }
    }
  };

  // Jika iframe sudah selesai menulis (beberapa browser tidak trigger onload), jalankan manual setelah delay
  setTimeout(()=>{ 
    if(iframe && iframe.parentNode){
      try{
        iframe.contentWindow.focus();
        iframe.contentWindow.print();
        setTimeout(()=>{ if(iframe && iframe.parentNode) iframe.parentNode.removeChild(iframe); }, 500);
      }catch(e){}
    }
  }, 1000);
}

/* helper untuk escape HTML agar aman di header cetak */
function escapeHtml(text){
  if(!text) return '';
  return text.replace(/&/g,'&amp;')
             .replace(/</g,'&lt;')
             .replace(/>/g,'&gt;')
             .replace(/"/g,'&quot;')
             .replace(/'/g,'&#039;');
}
</script>
</body>
</html>
